/***************************************************************************

  Sony ZN1/ZN2 - Arcade PSX Hardware
  ==================================
  Driver by smf

  QSound emulation based on information from the cps1/cps2 driver, Amuse & Miguel Angel Horna
  Taito FX1a sound emulation based on information from the Taito F2 driver

  The ZN1/ZN2 boards are standard boards from Sony. Extra sound h/w is
  included on the rom board,

  The BIOS is protected in all the games that run on this hardware.
  During boot up the BIOS decrypts parts of itself into ram using
  a device connected to the PSX controller port. As this is not
  emulated yet you only get the boot up colours.


  Capcom ZN1/ZN2
  --------------

  The QSound hardware is different to cps2 as it uses an i/o port and
  nmi's instead of shared memory. The driver uses 8bit i/o addresses
  but the real ZN1 hardware may use 16bit i/o addresses as the code
  always accesses port 0xf100. The ZN2 code however seems to vary the
  top eight bits of the address, this may or may not be important but
  it is currently ignored.

  The Gallop Racer dump came from a Capcom ZN1 QSound board but there
  were no QSound program/sample roms & the game doesn't appear to use
  it. Tecmo went on to produce games on their own boards.

    CP09 - kikaioh rom board
    CP10 - kikaioh/shiryu2 motherboard
    CP13 - shiryu2 rom board


  Taito FX1a
  ----------

  The extra sound h/w is the same as taito f2 ( z80-a + ym2610b ).

    TT02 - Super Football Champ
    TT06 - Magical Date


  Taito FX1b
  ----------

  The extra sound h/w isn't emulated and consists of an mn1020012a +
  zoom zsg-2 + tms57002dpha.

  mn1020012a is a 16bit panasonic mpu & the zoom zsg-2 is a sound generator
  lsi produced by zoom corp in 1995.

    TT04 - Ray Storm
    TT05 - Fighter's Impact Ace
    TT07 - G-Darius

  Beastorizer runs on a ray storm motherboard, but doesn't have fx1b sound h/w.

  Generic ZN1
  -----------

  None of these have extra sound hardware.

  Video System:
    KN01 - Motherboard
    KN02 - Sonic Wings Limited

  Tecmo:
    MG01 - Motherboard
    MG02 - Gallop Racer 2
    MG05 - Dead or Alive ++
    MG09 - Tondemo Crisis

***************************************************************************/

#define PROTECTION_HACK ( 1 )

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/mips/mips.h"
#include "cpu/z80/z80.h"
#include "sndhrdw/taitosnd.h"
#include "includes/psx.h"

#if PROTECTION_HACK

/* decodebp.c */
static unsigned char kn_protection_hack[] = 
{
	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* first test, this byte is ignored */
	0xef,

	/* checked for explicitly ( skips red screen ) */
	0x76, 0x00, 0xe3, 0xeb, 0x5b, 0x2f, 0x73, 0x56,
	0xc5, 0xc9, 0x74, 0x66, 0x79, 0xce, 0xa7, 0x02,
	0x77, 0x79, 0x2e, 0xbb, 0xda, 0x2c, 0xb3, 0x5b,
	0x65, 0xdc, 0xb7, 0x5b, 0x08, 0x14, 0xf1, 0xe4,
	0x73, 0x0e, 0x57, 0xc1, 0x29, 0x62, 0x76, 0x95,
	0xb9, 0x22, 0x3b, 0x7f, 0x28, 0x66, 0x32, 0xdd,
	0xcd,

	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	
	/* second test, this byte is ignored */
	0x00,

	/* checks the result of a calculation ( skips blue screen ) */
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* default value */
	0x00,
};

/* decodesce.c */
static unsigned char mg_protection_hack[] = 
{
	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* first test, this byte is ignored */
	0x6f,

	/* checked for explicitly ( skips red screen ) */
	0x5b, 0x35, 0xda, 0xf5, 0x42, 0x84, 0x84, 0xed,
	0x6f, 0x1f, 0x80, 0x99, 0x67, 0x1b, 0xcc, 0xa4,
	0x6c, 0x77, 0x8a, 0xad, 0xf7, 0xb6, 0xfa, 0xa7,
	0x36, 0x94, 0xcd, 0x64, 0x87, 0x7c, 0x52, 0x6c,
	0x76, 0x2a, 0x5e, 0x84, 0x7c, 0xa2, 0x7f, 0xf2,
	0xe5, 0xff, 0x56, 0x9b, 0x00, 0xb5, 0xf9, 0x69,
	0xdb,

	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* second test, this byte is ignored */
	0x00,

	/* checks the result of a calculation ( skips blue screen ) */
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* default value */
	0x00,
};

/* decodetaito.c */
static unsigned char tt_protection_hack[] = 
{
	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* first test, this byte is ignored */
	0xef,

	/* checked for explicitly ( skips red screen ) */
	0xd7, 0x5f, 0xb7, 0xea, 0x51, 0x15, 0x96, 0x02,
	0x0b, 0xf8, 0xf5, 0x54, 0xc5, 0xf9, 0x5d, 0x5d,
	0xea, 0x13, 0xd0, 0x63, 0x95, 0xd4, 0x9f, 0xee,
	0xeb, 0x3f, 0x94, 0x36, 0x42, 0x45, 0x81, 0xb0,
	0x80, 0xc4, 0x3f, 0x42, 0x39, 0x2a, 0xa4, 0x28,
	0x28, 0x20, 0xac, 0x00, 0x20, 0x8a, 0x28, 0x92,
	0xa8,

	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* second test, this byte is ignored */
	0x00,

	/* checks the result of a calculation ( skips blue screen ) */
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* default value */
	0x00,
};

/* decodecapcon.c */
static unsigned char cpzn1_protection_hack [] =
{
	
	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* first test, this byte is ignored */
	0xef,

	/* checked for explicitly ( skips red screen ) */
	0xe3, 0x9a, 0x29, 0xba, 0x16, 0x09, 0x82, 0x64,
	0x8b, 0xad, 0xf6, 0xfb, 0xbb, 0xba, 0x93, 0x3f,
	0xec, 0xc4, 0xe4, 0xad, 0xbe, 0xaf, 0x7a, 0x9e,
	0x7f, 0x79, 0xba, 0xac, 0xb6, 0x59, 0x11, 0x25,
	0xf3, 0x98, 0xa9, 0x57, 0x4d, 0xe0, 0xf6, 0x74,
	0xc4, 0x2c, 0x2b, 0xa0, 0x28, 0xee, 0xce, 0x83,
	0xc5,

	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* second test, this byte is ignored */
	0x00,

	/* checks the result of a calculation ( skips blue screen ) */
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* default value */
	0x00,
};

/* decodecapcon.c */
static unsigned char cpzn2_protection_hack [] =
{
	
	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* first test, this byte is ignored */
	0xef,

	/* checked for explicitly ( skips red screen ) */
	0xa7, 0x5d, 0x52, 0x7c, 0xdb, 0x26, 0x51, 0x1c,
	0x9d, 0x64, 0x87, 0xdd, 0x36, 0x25, 0xf1, 0xff,
	0xc4, 0xb7, 0x6d, 0xa7, 0x19, 0xca, 0xa2, 0xc3,
	0x76, 0x92, 0xcb, 0x64, 0xab, 0xc1, 0xae, 0xd1,
	0xda, 0x90, 0x90, 0x20, 0xfd, 0x2f, 0x93, 0x48,
	0x7e, 0xaf, 0x35, 0x82, 0xd4, 0x26, 0x6d, 0x76,
	0xa3,

	/* unknown */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* second test, this byte is ignored */
	0x00,

	/* checks the result of a calculation ( skips blue screen ) */
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00,

	/* default value */
	0x00,
};

static unsigned short n_sio0_ptr;
static unsigned char *p_sio0_ret;
static unsigned short n_sio0_ret_size;

#endif

static READ32_HANDLER( sio0_r )
{
	unsigned short n_return;

	n_return = 0;
	switch( offset )
	{
	case 0x00:
		/* data */
#if PROTECTION_HACK
		if( p_sio0_ret != NULL )
		{
			n_return = p_sio0_ret[ n_sio0_ptr ];
			if( n_sio0_ptr < n_sio0_ret_size - 1 )
			{
				n_sio0_ptr++;
			}
		}
#endif
		break;
	case 0x01:
		/* status */
#if PROTECTION_HACK
		if( p_sio0_ret != NULL )
		{
			n_return = 2;
		}
#endif
		break;
	case 0x02:
		/* mode */
		break;
	case 0x03:
		/* control */
		break;
	case 0x04:
		/* baud */
		break;
	}
	logerror( "%08x: sio0_r( %04x ) %04x\n", activecpu_get_pc(), offset, n_return );
	return n_return;
}

static WRITE32_HANDLER( sio0_w )
{
	int n_char;
	char s_char[ 5 ];

	s_char[ 0 ] = 0;
	switch( offset )
	{
	case 0x00:
		/* data */
		n_char = ( data & 0xff );
		if( n_char >= 0x20 && n_char <= 0x7f )
		{
			sprintf( s_char, " '%c'", n_char );
		}
		else
		{
			sprintf( s_char, " $%02x", n_char );
		}
		break;
	case 0x01:
		/* status */
		break;
	case 0x02:
		/* mode */
		break;
	case 0x03:
		/* control */
		break;
	case 0x04:
		/* baud */
		break;
	}
	logerror( "%08x: sio0_w( %04x, %08x )%s\n", activecpu_get_pc(), offset, data, s_char );
}

static WRITE_HANDLER( qsound_bankswitch_w )
{
	cpu_setbank( 10, memory_region( REGION_CPU2 ) + 0x10000 + ( ( data & 0x0f ) * 0x4000 ) );
}

static WRITE_HANDLER( fx1a_sound_bankswitch_w )
{
	cpu_setbank( 10, memory_region( REGION_CPU2 ) + 0x10000 + ( ( ( data - 1 ) & 0x07 ) * 0x4000 ) );
}

static MEMORY_READ32_START( zn_readmem )
	{ 0x00000000, 0x003fffff, MRA32_RAM },		/* ram */
	{ 0x1f000000, 0x1f3fffff, MRA32_BANK1 },	/* game rom */
	{ 0x1f800000, 0x1f8003ff, MRA32_BANK3 },	/* scratchpad */
	{ 0x1f801040, 0x1f80104f, sio0_r },
	{ 0x1f801810, 0x1f801817, psxgpu_r },
	{ 0x1fb80000, 0x1fbbffff, MRA32_BANK6 },	/* country rom */
	{ 0x80000000, 0x803fffff, MRA32_BANK4 },	/* ram mirror */
	{ 0xa0000000, 0xa03fffff, MRA32_BANK5 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MRA32_BANK7 },	/* bios */
MEMORY_END

static MEMORY_WRITE32_START( zn_writemem )
	{ 0x00000000, 0x003fffff, MWA32_RAM },		/* ram */
	{ 0x1f000000, 0x1f3fffff, MWA32_ROM },		/* game rom */
	{ 0x1f800000, 0x1f8003ff, MWA32_BANK3 },	/* scratchpad */
	{ 0x1f801040, 0x1f80104f, sio0_w },
	{ 0x1f801810, 0x1f801817, psxgpu_w },
	{ 0x1fb80000, 0x1fbbffff, MWA32_ROM },		/* country rom */
	{ 0x80000000, 0x803fffff, MWA32_BANK4 },	/* ram mirror */
	{ 0xa0000000, 0xa03fffff, MWA32_BANK5 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MWA32_ROM },		/* bios */
MEMORY_END

static MEMORY_READ_START( qsound_readmem )
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0xbfff, MRA_BANK10 },	/* banked (contains music data) */
	{ 0xd007, 0xd007, qsound_status_r },
	{ 0xf000, 0xffff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( qsound_writemem )
	{ 0x0000, 0xbfff, MWA_ROM },
	{ 0xd000, 0xd000, qsound_data_h_w },
	{ 0xd001, 0xd001, qsound_data_l_w },
	{ 0xd002, 0xd002, qsound_cmd_w },
	{ 0xd003, 0xd003, qsound_bankswitch_w },
	{ 0xf000, 0xffff, MWA_RAM },
MEMORY_END

static PORT_READ_START( qsound_readport )
	{ 0x00, 0x00, soundlatch_r },
PORT_END

static MEMORY_READ_START( link_readmem )
MEMORY_END

static MEMORY_WRITE_START( link_writemem )
MEMORY_END

static MEMORY_READ_START( fx1a_sound_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x7fff, MRA_BANK10 },
	{ 0xc000, 0xdfff, MRA_RAM },
	{ 0xe000, 0xe000, YM2610_status_port_0_A_r },
	{ 0xe001, 0xe001, YM2610_read_port_0_r },
	{ 0xe002, 0xe002, YM2610_status_port_0_B_r },
	{ 0xe200, 0xe200, MRA_NOP },
	{ 0xe201, 0xe201, taitosound_slave_comm_r },
	{ 0xea00, 0xea00, MRA_NOP },
MEMORY_END

static MEMORY_WRITE_START( fx1a_sound_writemem )
	{ 0x0000, 0x7fff, MWA_ROM },
	{ 0xc000, 0xdfff, MWA_RAM },
	{ 0xe000, 0xe000, YM2610_control_port_0_A_w },
	{ 0xe001, 0xe001, YM2610_data_port_0_A_w },
	{ 0xe002, 0xe002, YM2610_control_port_0_B_w },
	{ 0xe003, 0xe003, YM2610_data_port_0_B_w },
	{ 0xe200, 0xe200, taitosound_slave_port_w },
	{ 0xe201, 0xe201, taitosound_slave_comm_w },
	{ 0xe400, 0xe403, MWA_NOP }, /* pan */
	{ 0xee00, 0xee00, MWA_NOP }, /* ? */
	{ 0xf000, 0xf000, MWA_NOP }, /* ? */
	{ 0xf200, 0xf200, fx1a_sound_bankswitch_w },
MEMORY_END

static DRIVER_INIT( zn )
{
	cpu_setbank( 1, memory_region( REGION_USER1 ) ); /* game rom */
	cpu_setbank( 3, memory_region( REGION_CPU1 ) + 0x0480000 ); /* scratch pad */
	cpu_setbank( 4, memory_region( REGION_CPU1 ) ); /* ram mirror */
	cpu_setbank( 5, memory_region( REGION_CPU1 ) ); /* ram mirror */
	cpu_setbank( 6, memory_region( REGION_USER2 ) ); /* capcom country rom */
	cpu_setbank( 7, memory_region( REGION_CPU1 ) + 0x0400000 ); /* bios */

	if( strcmp( Machine->gamedrv->name, "glpracr" ) == 0 ||
		strcmp( Machine->gamedrv->name, "glprac2l" ) == 0 )
	{
		/* disable:
			the QSound CPU for glpracr as it doesn't have any roms &
			the link cpu for glprac2l as the h/w is not emulated yet. */
		timer_suspendcpu( 1, 1, SUSPEND_REASON_DISABLE );
	}

#if PROTECTION_HACK
	if( strcmp( Machine->gamedrv->name, "doapp" ) == 0 ||
		strcmp( Machine->gamedrv->name, "glpracr2" ) == 0 ||
		strcmp( Machine->gamedrv->name, "glprac2l" ) == 0 ||
		strcmp( Machine->gamedrv->name, "tondemo" ) == 0 ||
		strcmp( Machine->gamedrv->name, "brvblade" ) == 0 )
	{
		p_sio0_ret = mg_protection_hack;
		n_sio0_ret_size = sizeof( mg_protection_hack );
	}
	else if( strcmp( Machine->gamedrv->name, "sncwgltd" ) == 0 )
	{
		p_sio0_ret = kn_protection_hack;
		n_sio0_ret_size = sizeof( kn_protection_hack );
	}
	else if( strcmp( Machine->gamedrv->name, "ftimpcta" ) == 0 ||
		strcmp( Machine->gamedrv->name, "gdarius" ) == 0 ||
		strcmp( Machine->gamedrv->name, "gdarius2" ) == 0 ||
		strcmp( Machine->gamedrv->name, "mgcldate" ) == 0 ||
		strcmp( Machine->gamedrv->name, "psyforce" ) == 0 ||
		strcmp( Machine->gamedrv->name, "raystorm" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfchamp" ) == 0 ||
		strcmp( Machine->gamedrv->name, "beastrzb" ) == 0 )
	{
		p_sio0_ret = tt_protection_hack;
		n_sio0_ret_size = sizeof( tt_protection_hack );
	}
	else if( strcmp( Machine->gamedrv->name, "ts2j" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfex" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfexj" ) == 0 ||
		strcmp( Machine->gamedrv->name, "glpracr" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfexp" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfexpj" ) == 0 ||
		strcmp( Machine->gamedrv->name, "rvschool" ) == 0 ||
		strcmp( Machine->gamedrv->name, "jgakuen" ) == 0 ||
		strcmp( Machine->gamedrv->name, "tgmj" ) == 0 )
	{
		p_sio0_ret = cpzn1_protection_hack;
		n_sio0_ret_size = sizeof( cpzn1_protection_hack );
	}
	else if( strcmp( Machine->gamedrv->name, "sfex2" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sg2j" ) == 0 ||
		strcmp( Machine->gamedrv->name, "kikaioh" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfex2p" ) == 0 ||
		strcmp( Machine->gamedrv->name, "shiryu2" ) == 0 )
	{
		p_sio0_ret = cpzn2_protection_hack;
		n_sio0_ret_size = sizeof( cpzn2_protection_hack );
	}
	else
	{
		p_sio0_ret = NULL;
		n_sio0_ret_size = 0;
	}
#endif
}

static DRIVER_INIT( fx1b )
{
	init_zn();
	cpu_setbank( 10, memory_region( REGION_CPU2 ) );
}

/* sound player */

static int scode;
static int scode_last;
static int queue_data;
static int queue_len;
static int n_playermode;

static WRITE32_HANDLER( player_queue_w )
{
	if( cpu_getstatus( 1 ) != 0 )
	{
		queue_data = data;
		queue_len = 4;
	}
}

static void player_reset( void )
{
#if PROTECTION_HACK
	n_sio0_ptr = 0;
#endif
	queue_len = 0;
	scode_last = -1;

	if( strcmp( Machine->gamedrv->name, "sfex2" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfex2p" ) == 0 ||
		strcmp( Machine->gamedrv->name, "tgmj" ) == 0 )
	{
		scode = 0x0400;
	}
	else if( strcmp( Machine->gamedrv->name, "kikaioh" ) == 0 )
	{
		scode = 0x8000;
	}
	else
	{
		scode = 0x0000;
	}
	cpu_set_reset_line( 0, PULSE_LINE );
	cpu_set_reset_line( 1, PULSE_LINE );
	if( n_playermode == 0 )
	{
		timer_suspendcpu( 0, 0, SUSPEND_ANY_REASON );
	}
	else
	{
		timer_suspendcpu( 0, 1, SUSPEND_REASON_DISABLE );
	}
}

static VIDEO_UPDATE( player )
{
	if( keyboard_pressed_memory( KEYCODE_F1 ) )
	{
		n_playermode = !n_playermode;
		player_reset();
	}

	if( n_playermode == 0 )
	{
		video_update_psxgpu( bitmap, cliprect );
	}
	else
	{
		struct DisplayText dt[ 4 ];
		char text1[ 256 ];
		char text2[ 256 ];
		char text3[ 256 ];

		if( queue_len == 0 )
		{
			int stick;
			static int old_stick = 0x0f;

			stick = ~readinputport( 4 );
			if( ( stick & old_stick & 0x08 ) != 0 )
			{
				scode=( scode & 0xff00 ) | ( ( scode + 0x0001 ) & 0xff );
			}
			if( ( stick & old_stick & 0x04 ) != 0 )
			{
				scode=( scode & 0xff00 ) | ( ( scode - 0x0001 ) & 0xff );
			}
			if( ( stick & old_stick & 0x01 ) != 0 )
			{
				scode=( ( scode + 0x0100 ) & 0xff00 ) | ( scode & 0xff );
			}
			if( ( stick & old_stick & 0x02 ) != 0 )
			{
				scode=( ( scode - 0x0100 ) & 0xff00 ) | ( scode & 0xff );
			}
			old_stick = ~stick;
		}

		if( scode != scode_last )
		{
			player_queue_w( 0, scode, 0x0000ffff );
			scode_last = scode;
		}

		fillbitmap( bitmap, 0, &Machine->visible_area );

		sprintf( text1, "%s", Machine->gamedrv->description );
		if( strlen( text1 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text1[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		sprintf( text2, "SOUND CODE=%02x/%02x", scode >> 8, scode & 0xff );
		if( strlen( text2 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text2[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		sprintf( text3, "SELECT WITH RIGHT&LEFT/UP&DN" );
		if( strlen( text3 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text3[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		dt[ 0 ].text = text1;
		dt[ 0 ].color = UI_COLOR_NORMAL;
		dt[ 0 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 0 ].text ) ) / 2;
		dt[ 0 ].y = Machine->uiheight - Machine->uifontheight * 5;
		dt[ 1 ].text = text2;
		dt[ 1 ].color = UI_COLOR_NORMAL;
		dt[ 1 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 1 ].text ) ) / 2;
		dt[ 1 ].y = Machine->uiheight - Machine->uifontheight * 3;
		dt[ 2 ].text = text3;
		dt[ 2 ].color = UI_COLOR_NORMAL;
		dt[ 2 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 2 ].text ) ) / 2;
		dt[ 2 ].y = Machine->uiheight - Machine->uifontheight * 1;
		dt[ 3 ].text = 0; /* terminate array */
		displaytext( Machine->scrbitmap, dt );
	}
}

static MACHINE_INIT( zn )
{
	machine_init_psx();
	n_playermode = 0;
	player_reset();
}

static INTERRUPT_GEN( qsound_interrupt )
{
	if( queue_len == 4 )
	{
		soundlatch_w( 0, queue_data >> 8 );
		queue_len -= 2;
		cpu_set_irq_line(1, IRQ_LINE_NMI, PULSE_LINE);
	}
	else if( queue_len == 2 )
	{
		soundlatch_w( 0, queue_data & 0xff );
		queue_len -= 2;
		cpu_set_irq_line(1, IRQ_LINE_NMI, PULSE_LINE);
	}
	else
	{
		cpu_set_irq_line(1, 0, HOLD_LINE);
	}
}

static INTERRUPT_GEN( fx1a_sound_interrupt )
{
	if( queue_len == 4 )
	{
		taitosound_port_w( 0, 0 );
		taitosound_comm_w( 0, ( queue_data >> 0 ) & 0x0f );
		queue_len--;
	}
	else if( queue_len == 3 )
	{
		taitosound_port_w( 0, 1 );
		taitosound_comm_w( 0, ( queue_data >> 4 ) & 0x0f );
		queue_len--;
	}
	if( queue_len == 2 )
	{
		taitosound_port_w( 0, 2 );
		taitosound_comm_w( 0, ( queue_data >> 8 ) & 0x0f );
		queue_len--;
	}
	else if( queue_len == 1 )
	{
		taitosound_port_w( 0, 3 );
		taitosound_comm_w( 0, ( queue_data >> 12 ) & 0x0f );
		queue_len--;
	}
}

static struct QSound_interface qsound_interface =
{
	QSOUND_CLOCK,
	REGION_SOUND1,
	{ 100,100 }
};

/* handler called by the YM2610 emulator when the internal timers cause an IRQ */
static void irq_handler(int irq)
{
	cpu_set_irq_line(1,0,irq ? ASSERT_LINE : CLEAR_LINE);
}

static struct YM2610interface ym2610_interface =
{
	1,	/* 1 chip */
	16000000/2,	/* 8 MHz */
	{ 25 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ irq_handler },
	{ REGION_SOUND1 },	/* Delta-T */
	{ REGION_SOUND1 },	/* ADPCM */
	{ YM3012_VOL(100,MIXER_PAN_LEFT,100,MIXER_PAN_RIGHT) }
};

static MACHINE_DRIVER_START( zn )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33868800) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(psx)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 479, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(psxgpu)
	MDRV_VIDEO_START(psxgpu1024x1024)
	MDRV_VIDEO_UPDATE(psxgpu)
	MDRV_VIDEO_STOP(psxgpu)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( znqsound )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_CPU_ADD(Z80, 8000000)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)  /* 8MHz ?? */
	MDRV_CPU_MEMORY(qsound_readmem,qsound_writemem)
	MDRV_CPU_PORTS(qsound_readport,0)
	MDRV_CPU_VBLANK_INT(qsound_interrupt,4) /* 4 interrupts per frame ?? */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 479, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(psxgpu)
	MDRV_VIDEO_START(psxgpu1024x1024)
	MDRV_VIDEO_UPDATE(player)
	MDRV_VIDEO_STOP(psxgpu)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(QSOUND, qsound_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( znlink )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_CPU_ADD(Z80, 8000000)
	MDRV_CPU_MEMORY(link_readmem,link_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(psx)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 479, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(psxgpu)
	MDRV_VIDEO_START(psxgpu1024x1024)
	MDRV_VIDEO_UPDATE(psxgpu)
	MDRV_VIDEO_STOP(psxgpu)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( fx1a )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_CPU_ADD(Z80,16000000/4)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)	/* 4 MHz */
	MDRV_CPU_MEMORY(fx1a_sound_readmem,fx1a_sound_writemem)
	MDRV_CPU_VBLANK_INT(fx1a_sound_interrupt,1) /* 4 interrupts per frame ?? */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 479, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(psxgpu)
	MDRV_VIDEO_START(psxgpu1024x1024)
	MDRV_VIDEO_UPDATE(player)
	MDRV_VIDEO_STOP(psxgpu)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2610, ym2610_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( fx1b )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(psx)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 479, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(psxgpu)
	MDRV_VIDEO_START(psxgpu1024x1024)
	MDRV_VIDEO_UPDATE(psxgpu)
	MDRV_VIDEO_STOP(psxgpu)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

INPUT_PORTS_START( zn )
	PORT_START		/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	/* pause */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE	)	/* pause */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2  )

	PORT_START		/* DSWA */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* DSWB */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* DSWC */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* Player 1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )

	PORT_START		/* Player 2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
INPUT_PORTS_END


/* Capcom ZN1 */

#define CPZN1_BIOS \
	ROM_REGION( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "zn1.bin", 0x0400000, 0x080000, 0x50033af6 )

ROM_START( cpzn1 )
	CPZN1_BIOS
ROM_END

ROM_START( glpracr )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "graj-04.2j", 0x0000000, 0x080000, 0x53bf551c )

	ROM_REGION32_LE( 0x0c00000, REGION_USER1, 0 )
	ROM_LOAD( "gra-05m.3j", 0x0000000, 0x400000, 0x78053700 )
	ROM_LOAD( "gra-06m.4j", 0x0400000, 0x400000, 0xd73b392b )
	ROM_LOAD( "gra-07m.5j", 0x0800000, 0x400000, 0xacaefe3a )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
/* there are no QSound program roms
	ROM_LOAD( "gra-02",  0x00000, 0x08000, NO_DUMP )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "gra-03",  0x28000, 0x20000, NO_DUMP )
*/
	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
/* or QSound sample roms either
	ROM_LOAD( "gra-01m", 0x0000000, 0x400000, NO_DUMP )
*/
ROM_END

ROM_START( rvschool )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "jst-04a", 0x0000000, 0x080000, 0x034b1011 )

	ROM_REGION32_LE( 0x2400000, REGION_USER1, 0 )
	ROM_LOAD( "jst-05m", 0x0000000, 0x400000, 0x723372b8 )
	ROM_LOAD( "jst-06m", 0x0400000, 0x400000, 0x4248988e )
	ROM_LOAD( "jst-07m", 0x0800000, 0x400000, 0xc84c5a16 )
	ROM_LOAD( "jst-08m", 0x0c00000, 0x400000, 0x791b57f3 )
	ROM_LOAD( "jst-09m", 0x1000000, 0x400000, 0x6df42048 )
	ROM_LOAD( "jst-10m", 0x1400000, 0x400000, 0xd7e22769 )
	ROM_LOAD( "jst-11m", 0x1800000, 0x400000, 0x0a033ac5 )
	ROM_LOAD( "jst-12m", 0x1c00000, 0x400000, 0x43bd2ddd )
	ROM_LOAD( "jst-13m", 0x2000000, 0x400000, 0x6b443235 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "jst-02",  0x00000, 0x08000, 0x7809e2c3 )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "jst-03",  0x28000, 0x20000, 0x860ff24d )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "jst-01m", 0x0000000, 0x400000, 0x9a7c98f9 )
ROM_END

ROM_START( jgakuen )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "jst-04j", 0x0000000, 0x080000, 0x28b8000a )

	ROM_REGION32_LE( 0x2400000, REGION_USER1, 0 )
	ROM_LOAD( "jst-05m", 0x0000000, 0x400000, 0x723372b8 )
	ROM_LOAD( "jst-06m", 0x0400000, 0x400000, 0x4248988e )
	ROM_LOAD( "jst-07m", 0x0800000, 0x400000, 0xc84c5a16 )
	ROM_LOAD( "jst-08m", 0x0c00000, 0x400000, 0x791b57f3 )
	ROM_LOAD( "jst-09m", 0x1000000, 0x400000, 0x6df42048 )
	ROM_LOAD( "jst-10m", 0x1400000, 0x400000, 0xd7e22769 )
	ROM_LOAD( "jst-11m", 0x1800000, 0x400000, 0x0a033ac5 )
	ROM_LOAD( "jst-12m", 0x1c00000, 0x400000, 0x43bd2ddd )
	ROM_LOAD( "jst-13m", 0x2000000, 0x400000, 0x6b443235 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "jst-02",  0x00000, 0x08000, 0x7809e2c3 )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "jst-03",  0x28000, 0x20000, 0x860ff24d )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "jst-01m", 0x0000000, 0x400000, 0x9a7c98f9 )
ROM_END

ROM_START( sfex )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sfe-04a", 0x0000000, 0x080000, 0x08247bd4 )

	ROM_REGION32_LE( 0x1800000, REGION_USER1, 0 )
	ROM_LOAD( "sfe-05m", 0x0000000, 0x400000, 0xeab781fe )
	ROM_LOAD( "sfe-06m", 0x0400000, 0x400000, 0x999de60c )
	ROM_LOAD( "sfe-07m", 0x0800000, 0x400000, 0x76117b0a )
	ROM_LOAD( "sfe-08m", 0x0c00000, 0x400000, 0xa36bbec5 )
	ROM_LOAD( "sfe-09m", 0x1000000, 0x400000, 0x62c424cc )
	ROM_LOAD( "sfe-10m", 0x1400000, 0x400000, 0x83791a8b )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexj )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sfe-04j", 0x0000000, 0x080000, 0xea100607 )

	ROM_REGION32_LE( 0x1800000, REGION_USER1, 0 )
	ROM_LOAD( "sfe-05m", 0x0000000, 0x400000, 0xeab781fe )
	ROM_LOAD( "sfe-06m", 0x0400000, 0x400000, 0x999de60c )
	ROM_LOAD( "sfe-07m", 0x0800000, 0x400000, 0x76117b0a )
	ROM_LOAD( "sfe-08m", 0x0c00000, 0x400000, 0xa36bbec5 )
	ROM_LOAD( "sfe-09m", 0x1000000, 0x400000, 0x62c424cc )
	ROM_LOAD( "sfe-10m", 0x1400000, 0x400000, 0x83791a8b )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexp )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sfp-04e", 0x0000000, 0x080000, 0x305e4ec0 )

	ROM_REGION32_LE( 0x1880000, REGION_USER1, 0 )
	ROM_LOAD( "sfp-05",  0x0000000, 0x400000, 0xac7dcc5e )
	ROM_LOAD( "sfp-06",  0x0400000, 0x400000, 0x1d504758 )
	ROM_LOAD( "sfp-07",  0x0800000, 0x400000, 0x0f585f30 )
	ROM_LOAD( "sfp-08",  0x0c00000, 0x400000, 0x65eabc61 )
	ROM_LOAD( "sfp-09",  0x1000000, 0x400000, 0x15f8b71e )
	ROM_LOAD( "sfp-10",  0x1400000, 0x400000, 0xc1ecf652 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexpj )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sfp-04j", 0x0000000, 0x080000, 0x18d043f5 )

	ROM_REGION32_LE( 0x1800000, REGION_USER1, 0 )
	ROM_LOAD( "sfp-05",  0x0000000, 0x400000, 0xac7dcc5e )
	ROM_LOAD( "sfp-06",  0x0400000, 0x400000, 0x1d504758 )
	ROM_LOAD( "sfp-07",  0x0800000, 0x400000, 0x0f585f30 )
	ROM_LOAD( "sfp-08",  0x0c00000, 0x400000, 0x65eabc61 )
	ROM_LOAD( "sfp-09",  0x1000000, 0x400000, 0x15f8b71e )
	ROM_LOAD( "sfp-10",  0x1400000, 0x400000, 0xc1ecf652 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END



ROM_START( ts2j )
	CPZN1_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "ts2j-04", 0x0000000, 0x080000, 0x4aba8c5e )

	ROM_REGION32_LE( 0x0e00000, REGION_USER1, 0 )
	ROM_LOAD( "ts2-05",  0x0000000, 0x400000, 0x7f4228e2 )
	ROM_LOAD( "ts2-06m", 0x0400000, 0x400000, 0xcd7e0a27 )
	ROM_LOAD( "ts2-08m", 0x0800000, 0x400000, 0xb1f7f115 )
	ROM_LOAD( "ts2-10",  0x0c00000, 0x200000, 0xad90679a )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ts2-02",  0x00000, 0x08000, 0x2f45c461 )
	ROM_CONTINUE(        0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ts2-01",  0x0000000, 0x400000, 0xd7a505e0 )
ROM_END

/* Capcom ZN2 */

#define CPZN2_BIOS \
	ROM_REGION32_LE( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "cp10bios.bin", 0x0400000, 0x080000, 0xe860ea8b )

ROM_START( cpzn2 )
	CPZN2_BIOS
ROM_END

ROM_START( tgmj )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "ate-04j", 0x0000000, 0x080000, 0xbb4bbb96 )

	ROM_REGION32_LE( 0x0800000, REGION_USER1, 0 )
	ROM_LOAD( "ate-05",  0x0000000, 0x400000, 0x50977f5a )
	ROM_LOAD( "ate-06",  0x0400000, 0x400000, 0x05973f16 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ate-02",  0x00000, 0x08000, 0xf4f6e82f )
	ROM_CONTINUE(        0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ate-01",  0x0000000, 0x400000, 0xa21c6521 )
ROM_END

ROM_START( kikaioh )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "kioj-04.bin", 0x0000000, 0x080000, 0x3a2a3bc8 )

	ROM_REGION32_LE( 0x3000000, REGION_USER1, 0 )
	ROM_LOAD( "kio-05m.bin", 0x0000000, 0x800000, 0x98e9eb24 )
	ROM_LOAD( "kio-06m.bin", 0x0800000, 0x800000, 0xbe8d7d73 )
	ROM_LOAD( "kio-07m.bin", 0x1000000, 0x800000, 0xffd81f18 )
	ROM_LOAD( "kio-08m.bin", 0x1800000, 0x800000, 0x17302226 )
	ROM_LOAD( "kio-09m.bin", 0x2000000, 0x800000, 0xa34f2119 )
	ROM_LOAD( "kio-10m.bin", 0x2800000, 0x800000, 0x7400037a )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "kio-02.bin",  0x00000, 0x08000, 0x174309b3 )
	ROM_CONTINUE(            0x10000, 0x18000 )
	ROM_LOAD( "kio-03.bin",  0x28000, 0x20000, 0x0b313ae5 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "kio-01m.bin", 0x0000000, 0x400000, 0x6dc5bd07 )
ROM_END

ROM_START( sfex2 )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "ex2j-04", 0x0000000, 0x080000, 0x5d603586 )

	ROM_REGION32_LE( 0x2400000, REGION_USER1, 0 )
	ROM_LOAD( "ex2-05m", 0x0000000, 0x800000, 0x78726b17 )
	ROM_LOAD( "ex2-06m", 0x0800000, 0x800000, 0xbe1075ed )
	ROM_LOAD( "ex2-07m", 0x1000000, 0x800000, 0x6496c6ed )
	ROM_LOAD( "ex2-08m", 0x1800000, 0x800000, 0x3194132e )
	ROM_LOAD( "ex2-09m", 0x2000000, 0x400000, 0x075ae585 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ex2-02",  0x00000, 0x08000, 0x9489875e )
	ROM_CONTINUE(        0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ex2-01m", 0x0000000, 0x400000, 0x14a5bb0e )
ROM_END

ROM_START( sfex2p )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sf2p-04", 0x0000000, 0x080000, 0xc6d0aea3 )

	ROM_REGION32_LE( 0x3000000, REGION_USER1, 0 )
	ROM_LOAD( "sf2p-05", 0x0000000, 0x800000, 0x4ee3110f )
	ROM_LOAD( "sf2p-06", 0x0800000, 0x800000, 0x4cd53a45 )
	ROM_LOAD( "sf2p-07", 0x1000000, 0x800000, 0x11207c2a )
	ROM_LOAD( "sf2p-08", 0x1800000, 0x800000, 0x3560c2cc )
	ROM_LOAD( "sf2p-09", 0x2000000, 0x800000, 0x344aa227 )
	ROM_LOAD( "sf2p-10", 0x2800000, 0x800000, 0x2eef5931 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sf2p-02", 0x00000, 0x08000, 0x3705de5e )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sf2p-03", 0x28000, 0x20000, 0x6ae828f6 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ex2-01m", 0x0000000, 0x400000, 0x14a5bb0e )
ROM_END

ROM_START( sg2j )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "sg2j-04", 0x0000000, 0x080000, 0xcf4ce6ac )

	ROM_REGION32_LE( 0x2400000, REGION_USER1, 0 )
	ROM_LOAD( "sg2-05m", 0x0000000, 0x800000, 0xf1759236 )
	ROM_LOAD( "sg2-06m", 0x0800000, 0x800000, 0x33de4f72 )
	ROM_LOAD( "sg2-07m", 0x1000000, 0x800000, 0x72f724ba )
	ROM_LOAD( "sg2-08m", 0x1800000, 0x800000, 0x9e169eee )
	ROM_LOAD( "sg2-09m", 0x2000000, 0x400000, 0x33f73d4c )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sg2-02",  0x00000, 0x08000, 0x415ee138 )
	ROM_CONTINUE(        0x10000, 0x18000 )
	ROM_LOAD( "sg2-03",  0x28000, 0x20000, 0x43806735 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sg2-01m", 0x0000000, 0x400000, 0x643ea27b )
ROM_END

ROM_START( shiryu2 )
	CPZN2_BIOS

	ROM_REGION32_LE( 0x80000, REGION_USER2, 0 )
	ROM_LOAD( "hr2j-04.bin", 0x0000000, 0x080000, 0x0824ee5f )

	ROM_REGION32_LE( 0x2c00000, REGION_USER1, 0 )
	ROM_LOAD( "hr2-05m.bin", 0x0000000, 0x800000, 0x18716fe8 )
	ROM_LOAD( "hr2-06m.bin", 0x0800000, 0x800000, 0x6f13b69c )
	ROM_LOAD( "hr2-07m.bin", 0x1000000, 0x800000, 0x3925701b )
	ROM_LOAD( "hr2-08m.bin", 0x1800000, 0x800000, 0xd844c0dc )
	ROM_LOAD( "hr2-09m.bin", 0x2000000, 0x800000, 0xcdd43e6b )
	ROM_LOAD( "hr2-10m.bin", 0x2800000, 0x400000, 0xd95b3f37 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "hr2-02.bin",  0x00000, 0x08000, 0xacd8d385 )
	ROM_CONTINUE(            0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "hr2-01m.bin", 0x0000000, 0x200000, 0x510a16d1 )
	ROM_RELOAD( 0x0200000, 0x200000 )
ROM_END


/* Tecmo */

#define TPS_BIOS \
	ROM_REGION( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "mg-bios.bin",  0x0400000, 0x080000, 0x69ffbcb4 )

ROM_START( tps )
	TPS_BIOS
ROM_END

ROM_START( doapp )
	TPS_BIOS

	ROM_REGION32_LE( 0x01a00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "doapp119.bin", 0x0000001, 0x100000, 0xbbe04cef )
	ROM_LOAD16_BYTE( "doapp120.bin", 0x0000000, 0x100000, 0xb614d7e6 )
	ROM_LOAD( "doapp-0.216",         0x0200000, 0x400000, 0xacc6c539 )
	ROM_LOAD( "doapp-1.217",         0x0600000, 0x400000, 0x14b961c4 )
	ROM_LOAD( "doapp-2.218",         0x0a00000, 0x400000, 0x134f698f )
	ROM_LOAD( "doapp-3.219",         0x0e00000, 0x400000, 0x1c6540f3 )
	ROM_LOAD( "doapp-4.220",         0x1200000, 0x400000, 0xf83bacf7 )
	ROM_LOAD( "doapp-5.221",         0x1600000, 0x400000, 0xe11e8b71 )
ROM_END

ROM_START( glpracr2 )
	TPS_BIOS

	ROM_REGION32_LE( 0x02200000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "1.119",        0x0000001, 0x100000, 0x0fe2d2df )
	ROM_LOAD16_BYTE( "2.120",        0x0000000, 0x100000, 0x8e3fb1c0 )
	ROM_LOAD( "gra2-0.217",          0x0200000, 0x400000, 0xa077ffa3 )
	ROM_LOAD( "gra2-1.218",          0x0600000, 0x400000, 0x28ce033c )
	ROM_LOAD( "gra2-2.219",          0x0a00000, 0x400000, 0x0c9cb7da )
	ROM_LOAD( "gra2-3.220",          0x0e00000, 0x400000, 0x264e3a0c )
	ROM_LOAD( "gra2-4.221",          0x1200000, 0x400000, 0x2b070307 )
	ROM_LOAD( "gra2-5.222",          0x1600000, 0x400000, 0x94a363c1 )
	ROM_LOAD( "gra2-6.223",          0x1a00000, 0x400000, 0x8c6b4c4c )
	ROM_LOAD( "gra2-7.323",          0x1e00000, 0x400000, 0x7dfb6c54 )
ROM_END

ROM_START( glprac2l )
	TPS_BIOS

	ROM_REGION32_LE( 0x02200000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "gra2b.119",    0x0000001, 0x100000, 0x43abee7c )
	ROM_LOAD16_BYTE( "gra2a.120",    0x0000000, 0x100000, 0xf60096d4 )
	ROM_LOAD( "gra2-0.217",          0x0200000, 0x400000, 0xa077ffa3 )
	ROM_LOAD( "gra2-1.218",          0x0600000, 0x400000, 0x28ce033c )
	ROM_LOAD( "gra2-2.219",          0x0a00000, 0x400000, 0x0c9cb7da )
	ROM_LOAD( "gra2-3.220",          0x0e00000, 0x400000, 0x264e3a0c )
	ROM_LOAD( "gra2-4.221",          0x1200000, 0x400000, 0x2b070307 )
	ROM_LOAD( "gra2-5.222",          0x1600000, 0x400000, 0x94a363c1 )
	ROM_LOAD( "gra2-6.223",          0x1a00000, 0x400000, 0x8c6b4c4c )
	ROM_LOAD( "gra2-7.323",          0x1e00000, 0x400000, 0x7dfb6c54 )

	ROM_REGION( 0x040000, REGION_CPU2, 0 )
	ROM_LOAD( "link3118.bin", 0x0000000, 0x040000, 0xa4d4761e )
ROM_END

ROM_START( tondemo )
	TPS_BIOS

	ROM_REGION32_LE( 0x01600000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "u0119.bin",    0x0000001, 0x100000, 0x5711e301 )
	ROM_LOAD16_BYTE( "u0120.bin",    0x0000000, 0x100000, 0x0b8312c6 )
	ROM_LOAD( "tca-0.217",           0x0200000, 0x400000, 0xef175910 )
	ROM_LOAD( "tca-1.218",           0x0600000, 0x400000, 0xc3474e8a )
	ROM_LOAD( "tca-2.219",           0x0a00000, 0x400000, 0x89b8e1a8 )
	ROM_LOAD( "tca-3.220",           0x0e00000, 0x400000, 0x4fcf8032 )
	ROM_LOAD( "tca-4.221",           0x1200000, 0x400000, 0xc9e23f25 )
ROM_END

ROM_START( brvblade )
	TPS_BIOS

	ROM_REGION32_LE( 0x0c00000, REGION_USER1, 0 )
	/* TSOP56 Flash EEPROMs are not dumped */
	ROM_LOAD( "tsop56.1",     0x0000000, 0x0400000, 0x00000000 )
	ROM_LOAD( "tsop56.2",     0x0400000, 0x0400000, 0xffffffff )
	ROM_LOAD( "rom1u028.bin", 0x0800000, 0x0200000, 0xcb7bc657 ) 
	ROM_LOAD( "rom2u336.bin", 0x0a00000, 0x0200000, 0x4709026a ) 
	ROM_LOAD( "spu1u412.bin", 0x0b00000, 0x0080000, 0x6408b5b2 ) 
	ROM_LOAD( "spu0u049.bin", 0x0b80000, 0x0080000, 0xc9df8ed9 ) 
ROM_END


/* video system */

#define KN_BIOS \
	ROM_REGION( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "kn-bios.bin",  0x0400000, 0x080000, 0x5ff165f3 )

ROM_START( sncwgltd )
	KN_BIOS

	ROM_REGION32_LE( 0x01a80000, REGION_USER1, 0 )
	ROM_LOAD( "ic5.bin",      0x0000000, 0x080000, 0x458f14aa )
	ROM_LOAD( "ic6.bin",      0x0080000, 0x080000, 0x8233dd1e )
	ROM_LOAD( "ic7.bin",      0x0100000, 0x080000, 0xdf5ba2f7 )
	ROM_LOAD( "ic8.bin",      0x0180000, 0x080000, 0xe8145f2b )
	ROM_LOAD( "ic9.bin",      0x0200000, 0x080000, 0x605c9370 )
	ROM_LOAD( "ic11.bin",     0x0280000, 0x400000, 0xa93f6fee )
	ROM_LOAD( "ic12.bin",     0x0680000, 0x400000, 0x9f584ef7 )
	ROM_LOAD( "ic13.bin",     0x0a80000, 0x400000, 0x652e9c78 )
	ROM_LOAD( "ic14.bin",     0x0e80000, 0x400000, 0xc4ef1424 )
	ROM_LOAD( "ic15.bin",     0x1280000, 0x400000, 0x2551d816 )
	ROM_LOAD( "ic16.bin",     0x1680000, 0x400000, 0x21b401bc )
ROM_END


/* Taito FX1a/FX1b */

#define TAITOFX1_BIOS \
	ROM_REGION( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "tt-bios.bin",  0x0400000, 0x080000, 0xe3f23b6e )

ROM_START( taitofx1 )
	TAITOFX1_BIOS
ROM_END

ROM_START( ftimpcta )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e25-14.3",     0x0000001, 0x100000, 0x0c5f474f )
	ROM_LOAD16_BYTE( "e25-13.4",     0x0000000, 0x100000, 0x7f078d7b )
	ROM_LOAD( "e25-01.1",            0x0200000, 0x400000, 0x8cc4be0c )
	ROM_LOAD( "e25-02.2",            0x0600000, 0x400000, 0x8e8b4c82 )
	ROM_LOAD( "e25-03.12",           0x0a00000, 0x400000, 0x43b1c085 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e25-10.14",    0x0000000, 0x080000, 0x2b2ad1b1 )

	ROM_REGION( 0x600000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e25-04.27",    0x0000000, 0x400000, 0x09a66d35 )
	ROM_LOAD( "e25-05.28",    0x0040000, 0x200000, 0x3fb57636 )
ROM_END

ROM_START( gdarius )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e39-05.3",     0x0000001, 0x100000, 0x750e5b13 )
	ROM_LOAD16_BYTE( "e39-06.4",     0x0000000, 0x100000, 0x2980c30d )
	ROM_LOAD( "e39-01.1",            0x0200000, 0x400000, 0xbdaaa251 )
	ROM_LOAD( "e39-02.2",            0x0600000, 0x400000, 0xa47aab5d )
	ROM_LOAD( "e39-03.12",           0x0a00000, 0x400000, 0xa883b6a5 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e39-07.14",    0x0000000, 0x080000, 0x2252c7c1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e39-04.27",    0x0000000, 0x400000, 0x6ee35e68 )
ROM_END

ROM_START( gdarius2 )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e39-11.3",     0x0000001, 0x100000, 0x766f73df )
	ROM_LOAD16_BYTE( "e39-12.4",     0x0000000, 0x100000, 0xb23266c3 )
	ROM_LOAD( "e39-01.1",            0x0200000, 0x400000, 0xbdaaa251 )
	ROM_LOAD( "e39-02.2",            0x0600000, 0x400000, 0xa47aab5d )
	ROM_LOAD( "e39-03.12",           0x0a00000, 0x400000, 0xa883b6a5 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e39-07.14",    0x0000000, 0x080000, 0x2252c7c1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e39-04.27",    0x0000000, 0x400000, 0x6ee35e68 )
ROM_END

ROM_START( mgcldate )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00c00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e32-08.2",     0x0000000, 0x100000, 0x3d42cd28 ) 
	ROM_LOAD16_BYTE( "e32-09.7",     0x0000001, 0x100000, 0xdb7ec115 ) 
	ROM_LOAD( "e32-01.1",            0x0200000, 0x400000, 0xcf5f1d01 )
	ROM_LOAD( "e32-02.6",            0x0600000, 0x400000, 0x61c8438c )
	ROM_LOAD( "e32-03.12",           0x0a00000, 0x200000, 0x190d1618 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e32-10.22",           0x0000000, 0x004000, 0xadf3feb5 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e32-04.15",           0x0000000, 0x400000, 0xc72f9eea )
ROM_END

ROM_START( mgcldtea )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00c00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e32-05.2",     0x0000000, 0x080000, 0x72fc7f7b )
	ROM_LOAD16_BYTE( "e32-06.7",     0x0000001, 0x080000, 0xd11c3881 )
	ROM_LOAD( "e32-01.1",            0x0200000, 0x400000, 0xcf5f1d01 )
	ROM_LOAD( "e32-02.6",            0x0600000, 0x400000, 0x61c8438c )
	ROM_LOAD( "e32-03.12",           0x0a00000, 0x200000, 0x190d1618 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e32-10.22",           0x0000000, 0x004000, 0xadf3feb5 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e32-04.15",           0x0000000, 0x400000, 0xc72f9eea )
ROM_END

ROM_START( psyforce )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e22-05.2",     0x0000000, 0x080000, 0x7770242c )
	ROM_LOAD16_BYTE( "e22-10.7",     0x0000001, 0x080000, 0xf6341d63 )
	ROM_LOAD( "e22-02.16",           0x0100000, 0x200000, 0x03b50064 )
	ROM_LOAD( "e22-03.1",            0x0300000, 0x200000, 0x8372f839 )
	ROM_LOAD( "e22-04.21",           0x0500000, 0x200000, 0x397b71aa )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e22-07.22",           0x0000000, 0x004000, 0x739af589 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x200000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e22-01.15",           0x000000,  0x200000, 0x808b8340 )
ROM_END

ROM_START( raystorm )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e24-06.3",     0x0000001, 0x080000, 0xd70cdf46 )
	ROM_LOAD16_BYTE( "e24-05.4",     0x0000000, 0x080000, 0x40097ab9 )
	ROM_LOAD( "e24-02.1",            0x0100000, 0x400000, 0x9f70950d )
	ROM_LOAD( "e24-03.2",            0x0500000, 0x400000, 0x6c1f0a5d )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e24-09.14",    0x0000000, 0x080000, 0x808589e1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e24-04.27",    0x0000000, 0x400000, 0xf403493a )
ROM_END

ROM_START( sfchamp )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e18-07.2",     0x0000000, 0x080000, 0x1b484e1c )
	ROM_LOAD16_BYTE( "e18-08.7",     0x0000001, 0x080000, 0x6a5558cd )
	ROM_LOAD( "e18-02.12",           0x0100000, 0x200000, 0xc7b4fe29 )
	ROM_LOAD( "e18-03.16",           0x0300000, 0x200000, 0x76392346 )
	ROM_LOAD( "e18-04.19",           0x0500000, 0x200000, 0xfc3731da )
	ROM_LOAD( "e18-05.21",           0x0700000, 0x200000, 0x2e984c50 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e18-09.22",           0x0000000, 0x004000, 0xbb5a5319 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x200000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e18-01.15",           0x0000000, 0x200000, 0xdbd1408c )
ROM_END

/* Eighteen Raizing */

ROM_START( beastrzb )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x0a80000, REGION_USER1, 0 )
	ROM_LOAD( "27c160.1",     0x0000000, 0x200000, 0x820855e2 )
	ROM_LOAD( "27c160.2",     0x0200000, 0x200000, 0x1712af34 )
	ROM_LOAD( "27c800.3",     0x0400000, 0x100000, 0x7192eb4e )
	ROM_LOAD( "4",            0x0500000, 0x200000, 0xbff21f44 )
	ROM_LOAD( "27c160.5",     0x0700000, 0x200000, 0x10fe6f4d )
	ROM_LOAD( "27c4096.1",    0x0900000, 0x080000, 0x217734a1 )
	ROM_LOAD( "27c4096.2",    0x0980000, 0x080000, 0xd1f2a9b2 )
	ROM_LOAD( "27c240.3",     0x0a00000, 0x080000, 0x509cdc8b )
ROM_END



/* Capcom ZN1 */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the zn1.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, cpzn1,    0,        zn,       zn, zn,   ROT0, "Sony/Capcom", "ZN1", NOT_A_DRIVER )

GAMEX( 1995, ts2j,     cpzn1,    znqsound, zn, zn,   ROT0, "Capcom/Takara", "Battle Arena Toshinden 2 (JAPAN 951124)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1996, sfex,     cpzn1,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX (ASIA 961219)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1996, sfexj,    sfex,     znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX (JAPAN 961130)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1996, glpracr,  cpzn1,    zn,       zn, zn,   ROT0, "Tecmo", "Gallop Racer (JAPAN Ver 9.01.12)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1997, sfexp,    cpzn1,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX Plus (USA 970311)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1997, sfexpj,   sfexp,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX Plus (JAPAN 970311)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1997, rvschool, cpzn1,    znqsound, zn, zn,   ROT0, "Capcom", "Rival Schools (ASIA 971117)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1997, jgakuen,  rvschool, znqsound, zn, zn,   ROT0, "Capcom", "Justice Gakuen (JAPAN 971117)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1998, tgmj,     cpzn1,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Tetris The Grand Master (JAPAN 980710)", GAME_UNEMULATED_PROTECTION )

/* Capcom ZN2 */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the cp10bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, cpzn2,    0,        zn,       zn, zn,   ROT0, "Sony/Capcom", "ZN2", NOT_A_DRIVER )

GAMEX( 1998, sfex2,    cpzn2,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX 2 (JAPAN 980312)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1998, sg2j,     cpzn2,    znqsound, zn, zn,   ROT0, "Capcom","Star Gladiator 2 (JAPAN 980316)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1998, kikaioh,  cpzn2,    znqsound, zn, zn,   ROT0, "Capcom", "Kikaioh (JAPAN 980914)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1999, sfex2p,   sfex2,    znqsound, zn, zn,   ROT0, "Capcom/Arika", "Street Fighter EX 2 Plus (JAPAN 990611)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1999, shiryu2,  cpzn2,    znqsound, zn, zn,   ROT0, "Capcom", "Strider Hiryu 2 (JAPAN 991213)", GAME_UNEMULATED_PROTECTION )

/* Tecmo */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the mg-bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, tps,      0,        zn,       zn, zn,   ROT0, "Sony/Tecmo", "TPS", NOT_A_DRIVER )

GAMEX( 1997, glpracr2, tps,      zn,       zn, zn,   ROT0, "Tecmo", "Gallop Racer 2 (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1997, glprac2l, glpracr2, znlink,   zn, zn,   ROT0, "Tecmo", "Gallop Racer 2 Link HW (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1998, doapp,    tps,      zn,       zn, zn,   ROT0, "Tecmo", "Dead Or Alive ++ (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1999, tondemo,  tps,      zn,       zn, zn,   ROT0, "Tecmo", "Tondemo Crisis (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 2000, brvblade, tps,      zn,       zn, zn,   ROT0, "Raizing", "Brave Blade (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )

/* Video System */
GAMEX( 1996, sncwgltd, 0,        zn,       zn, zn,   ROT0, "Video System", "Sonic Wings Limited (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )

/* Taito */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the tt-bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1995, taitofx1, 0,        fx1b,     zn, zn,   ROT0, "Sony/Taito", "Taito FX1", NOT_A_DRIVER )

GAMEX( 1995, psyforce, taitofx1, fx1a,     zn, zn,   ROT0, "Taito", "Psychic Force (JAPAN)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1995, sfchamp,  taitofx1, fx1a,     zn, zn,   ROT0, "Taito", "Super Football Champ (JAPAN)", GAME_UNEMULATED_PROTECTION )
GAMEX( 1996, raystorm, taitofx1, fx1b,     zn, fx1b, ROT0, "Taito", "Ray Storm (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1996, ftimpcta, taitofx1, fx1b,     zn, fx1b, ROT0, "Taito", "Fighter's Impact Ace (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1996, mgcldate, taitofx1, fx1a,     zn, zn,   ROT0, "Taito", "Magical Date (JAPAN) set 1", GAME_UNEMULATED_PROTECTION )
GAMEX( 1996, mgcldtea, mgcldate, fx1a,     zn, fx1b, ROT0, "Taito", "Magical Date (JAPAN) set 2", GAME_UNEMULATED_PROTECTION )
GAMEX( 1997, gdarius,  taitofx1, fx1b,     zn, fx1b, ROT0, "Taito", "G-Darius (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
GAMEX( 1997, gdarius2, gdarius,  fx1b,     zn, fx1b, ROT0, "Taito", "G-Darius Ver.2 (JAPAN)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )

/* Eighteen Raizing */
GAMEX( 1997, beastrzb, taitofx1, zn,       zn, zn,   ROT0, "Eighteen Raizing", "Beastorizer (bootleg)", GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND )
